/*
  PIC16F15224

  LED1 -> RA0
  LED2 -> RA5
  LED3 -> RC5
  LED4 -> RC3
  LED5 -> RC1
  LED6 -> RA4
  LED7 -> RC4
  LED8 -> RA1

  VR1  -> RA2/ANA2 (speed/direction)
  VR2  -> RC2/ANC2 (brightness/beam width)

  Use Timer0 for main loop (1ms rate)
  Use Timer1 and two compare modules for PWM
*/

#include "config.h"
#include <xc.h>
#include <string.h>///////////

#define PIN0 (1<<0)
#define PIN1 (1<<1)
#define PIN2 (1<<2)
#define PIN3 (1<<3)
#define PIN4 (1<<4)
#define PIN5 (1<<5)

unsigned char newLATA, newLATC, newPWM[4];
typedef struct { unsigned char A; unsigned char C; } ports;
static const ports LEDs[8] = { { PIN0, 0 }, { PIN5, 0 }, { 0, PIN5 }, { 0, PIN3 }, { 0, PIN1 }, { PIN4, 0 }, { 0, PIN4 }, { PIN1, 0 } };

void PinsInit() {
    // LED outputs: RA0, RA1, RA4, RA5, RC1, RC3, RC4, RC5 (initialise to high)
    // ADC inputs: RA2, RC2
    // unused (set as output): RC0
    newLATA = LATA = 0b00110011;
    newLATC = LATC = 0b00111010;
    ANSELA = 0b11001100;
    ANSELC = 0b11000101;
    TRISA = 0b00001100;
    TRISC = 0b00000100;
}

void Timer0Init() {
    // set up Timer0 for 8ms interval, setting T0IF
    PIR0 = 0;
    PIE0 = 0;
    TMR0H = 255;
    TMR0L = 0;
    T0CON1 = 0b01000011; // prescaler = 8
    T0CON0 = 0b11100000;
}

void Timer2Init() {
    T2CLKCON = 1; // Fosc/4
    T2HLT = 0b10100000;
    T2PR = 0xFF;
    T2CON = 0b10000000;
    PWM3CON = 0b10010000;
    PWM4CON = 0b10010000;
}

void ADCInit() {
    ADCON0=0;            //reset
    ADCON1= 0b10110000;  //1=right align, 011=ADCRC, 00=NA, 00=V+REF=VDD    
    ADACT=0;             //no auto trigger    
    ADCON0bits.ADON=1;   //turn on    
}

unsigned short ReadADC(unsigned char c){   //get result from selected channel
    ADCON0bits.CHS=c;
    ADCON0bits.ADGO=1;              //start a reading
    while(ADCON0bits.ADGO){}        //wait till done
    return ADRES;
}

void SetRxyPPS(unsigned char PWM1, unsigned char PWM2) {
    RA0PPS = (PWM1 == 1 ? 3 : (PWM2 == 1 ? 4 : 0));
    RA5PPS = (PWM1 == 2 ? 3 : (PWM2 == 2 ? 4 : 0));
    RC5PPS = (PWM1 == 3 ? 3 : (PWM2 == 3 ? 4 : 0));
    RC3PPS = (PWM1 == 4 ? 3 : (PWM2 == 4 ? 4 : 0));
    RC1PPS = (PWM1 == 5 ? 3 : (PWM2 == 5 ? 4 : 0));
    RA4PPS = (PWM1 == 6 ? 3 : (PWM2 == 6 ? 4 : 0));
    RC4PPS = (PWM1 == 7 ? 3 : (PWM2 == 7 ? 4 : 0));
    RA1PPS = (PWM1 == 8 ? 3 : (PWM2 == 8 ? 4 : 0));
}

void SetPWM1(unsigned char delay) {
    PWM3DCH = delay;
}

void SetPWM2(unsigned char delay) {
    PWM4DCH = delay;
}

void UpdateLEDs(unsigned char A, unsigned char C, unsigned char* PWM) {
    LATA = A;
    LATC = C;
    SetRxyPPS(PWM[0], PWM[1]);
    SetPWM1(PWM[2]);
    SetPWM2(PWM[3]);
}

unsigned char lights[8];
unsigned short dir = 512;
unsigned short spread = 512*32;
void update_lights() {
  unsigned char l;
  unsigned short ldir = 0;
  signed short diff, amt;
  for( l = 0; l < 8; ++l ) {
    diff = (signed short)(dir - ldir);
    if( diff < 0 )
      diff = -diff;
    if( diff >= 0 && diff < spread ) {
      if( spread > 255*32 && diff < spread-255*32 )
        lights[l] = 255;
      else
        lights[l] = (unsigned char)((spread-(unsigned short)diff)>>5);
    } else {
      lights[l] = 0;
    }
    ldir += 8192;
  }
}

void update_LEDs() {
  unsigned char A, C, i, PWM[4];
  A = 0xFF;
  C = 0xFF;
  PWM[0] = 0;
  PWM[1] = 0;
  PWM[2] = 0;
  PWM[3] = 0;
  for( i = 0; i < 8; ++i ) {
    if( lights[i] == 255 ) {
      A &= ~LEDs[i].A;
      C &= ~LEDs[i].C;
    }
    if( lights[i] > 0 && lights[i] < 255 && !PWM[0] ) {
      PWM[0] = i+1;
      PWM[2] = lights[i];
    } else if( lights[i] > 0 && lights[i] < 255 && !PWM[1] ) {
      PWM[1] = i+1;
      PWM[3] = lights[i];
    }
  }
  UpdateLEDs(A, C, PWM);
//  RA0PPS = 3;
//  PWM3DCH = 0x7F;
}

void main(void) {
    unsigned char counter = 0;
    unsigned short speed = 512+128;

    PinsInit();
    Timer0Init();
    Timer2Init();
//    ADCInit();

    while(1) {
        // wait for 8ms tick
        while( !PIR0bits.TMR0IF )
            ;
        PIR0bits.TMR0IF = 0;
        update_lights();
        update_LEDs();
        dir = dir + (speed-512);

//        if( ++counter == 16 ) {        
//            speed = (speed + ReadADC(2)) >> 1; // RA2
//            spread = (spread + ReadADC(18)) >> 1; // RC2
//            counter = 0;
//        }
    }
}
